# Hosting Next.js + Headless WordPress on cPanel

## Current Setup
- WordPress (Headless): `paulagonda.cybaxtech.com`
- Next.js App: Needs to be hosted and connected

## Best Configuration for Your Domain

### Option 1: Subdomain Approach (RECOMMENDED - EASIEST)
```
paulagonda.cybaxtech.com          → WordPress API (already hosted)
www.paulagonda.cybaxtech.com      → Next.js Application (new)
```

**Pros:**
- Cleanest separation of concerns
- Easy to manage and scale
- WordPress and Next.js run independently
- Simple to set up on cPanel
- No conflicts between the two

**Cons:**
- Users see www subdomain in URL

### Option 2: Same Domain with Path Separation
```
paulagonda.cybaxtech.com                  → Next.js Frontend (main app)
paulagonda.cybaxtech.com/wp-admin         → WordPress Admin
paulagonda.cybaxtech.com/wp-json/...      → WordPress API (already working)
```

**Pros:**
- Single domain for users
- Cleaner user experience

**Cons:**
- More complex cPanel configuration
- Requires reverse proxy setup
- Higher maintenance

### Option 3: Separate Subdomains (MOST PROFESSIONAL)
```
paulagonda.cybaxtech.com          → Next.js Frontend
api.paulagonda.cybaxtech.com      → WordPress API
wp.paulagonda.cybaxtech.com       → WordPress Admin
```

**Pros:**
- Most professional setup
- Clear separation of services
- Easy to scale independently
- Best for SEO and analytics

**Cons:**
- Requires more DNS setup
- Slightly more complex to manage

---

## RECOMMENDED: Option 1 (Subdomain) - Step-by-Step Setup

### Step 1: Check Your Hosting Provider's Node.js Support

**Important:** cPanel doesn't natively support Node.js. You need hosting that provides:
- Node.js runtime support
- npm/yarn package manager
- Port forwarding capabilities
- SSH access

**Check if your current hosting supports Node.js:**
1. Log into cPanel
2. Look for "Node.js" or "Ruby/Rails" in the sidebar
3. If not available, you'll need to upgrade your hosting plan

### Step 2: If Your Hosting Supports Node.js

#### 2.1 Create a Subdomain in cPanel
1. Go to cPanel Dashboard
2. Navigate to "Addon Domains" or "Subdomains"
3. Create subdomain: `www.paulagonda.cybaxtech.com`
4. Document the directory (usually `/public_html/www`)

#### 2.2 Upload Your Next.js Project
```bash
# On your local machine
# 1. Build the Next.js project
npm run build

# 2. Create a production-ready package
npm install --production

# 3. Upload to cPanel via SFTP/SSH
# - Host: paulagonda.cybaxtech.com
# - Username: your cPanel username
# - Password: your cPanel password
# - Upload to: /public_html/www or the directory from step 2.1

# Or use SCP if you have SSH
scp -r ./* username@paulagonda.cybaxtech.com:/home/username/public_html/www/
```

#### 2.3 Create Node.js App in cPanel
1. In cPanel, go to "Node.js"
2. Click "Create Application"
3. Configure:
   - **Node.js Version:** 18.x or newer
   - **Application Mode:** Production
   - **Application Root:** `/public_html/www`
   - **Application URL:** `www.paulagonda.cybaxtech.com`
   - **Application Startup File:** `server.js` (but Next.js uses `.next`)
   
   **Important:** For Next.js, you may need to use a custom startup approach

#### 2.4 Set Environment Variables in cPanel
1. In Node.js app settings, find "Environment Variables"
2. Add your WordPress URL:
```
NEXT_PUBLIC_WORDPRESS_URL=https://paulagonda.cybaxtech.com
```

#### 2.5 Install Dependencies
```bash
# Via cPanel Terminal or SSH
cd /home/username/public_html/www
npm install
npm run build
```

---

## BETTER ALTERNATIVE: Use a Specialized Node.js Hosting

Since cPanel is traditionally designed for PHP, consider these alternatives for better Next.js support:

### Option A: Vercel (EASIEST - FREE TIER AVAILABLE)
**Pros:**
- Built specifically for Next.js
- Free tier includes unlimited deployments
- Automatic HTTPS, CDN, and scaling
- Connects to your GitHub for automatic updates
- Environment variables managed in dashboard

**Steps:**
```
1. Push your code to GitHub
2. Visit vercel.com/new
3. Import your GitHub repository
4. Add environment variable: NEXT_PUBLIC_WORDPRESS_URL=https://paulagonda.cybaxtech.com
5. Deploy (automatic)
6. Connect custom domain: paulagonda.cybaxtech.com or www.paulagonda.cybaxtech.com
```

### Option B: DigitalOcean / Heroku
- Good Node.js support
- Custom domain support
- More control than Vercel
- Usually $5-12/month

### Option C: Keep WordPress on cPanel + Host Next.js Separately
This is the **RECOMMENDED PRODUCTION SETUP:**

```
cPanel Hosting (Current): paulagonda.cybaxtech.com (WordPress API)
Vercel or VPS: Next.js Frontend on same or subdomain
Both connected via REST API
```

---

## Configuration for Same Domain (With Reverse Proxy)

If you absolutely want the same domain, you need a reverse proxy:

### Using cPanel with Reverse Proxy (Advanced)

```
paulagonda.cybaxtech.com:80 (cPanel)
  ├─ /wp-admin, /wp-json → Routes to WordPress (PHP)
  └─ / (all other routes) → Routes to Node.js port (3000)
```

**cPanel Configuration:**
1. Create `.htaccess` file in `/public_html`:
```apache
<IfModule mod_rewrite.c>
  RewriteEngine On
  
  # Route WordPress admin and API to local PHP
  RewriteCond %{REQUEST_URI} ^/wp-admin [OR]
  RewriteCond %{REQUEST_URI} ^/wp-json [OR]
  RewriteCond %{REQUEST_URI} ^/wp-login
  RewriteCond %{REQUEST_FILENAME} -f [OR]
  RewriteCond %{REQUEST_FILENAME} -d
  RewritePass
  
  # Everything else goes to Node.js app running on port 3000
  RewriteRule ^(.*)$ http://localhost:3000/$1 [P,L]
</IfModule>
```

2. In cPanel Node.js app, set to run on port `3000`
3. Ensure Apache ProxyPass is enabled

**⚠️ Warning:** This is complex and requires technical knowledge. Not recommended for beginners.

---

## FINAL RECOMMENDATION

### Best Setup for Your Situation:

1. **Keep WordPress on cPanel** (paulagonda.cybaxtech.com) - Already working
2. **Host Next.js on Vercel** (free tier):
   - Push to GitHub
   - Deploy to Vercel
   - Configure NEXT_PUBLIC_WORDPRESS_URL in Vercel dashboard
   - Map domain: `www.paulagonda.cybaxtech.com` or `paulagonda.cybaxtech.com` to Vercel
3. **Update WordPress URL in Next.js environment:**
   - In Vercel Settings → Environment Variables:
   ```
   NEXT_PUBLIC_WORDPRESS_URL=https://paulagonda.cybaxtech.com
   ```

This approach:
- ✅ Minimal configuration
- ✅ Automatic scaling and HTTPS
- ✅ Free hosting (Vercel)
- ✅ No complex server setup
- ✅ Professional performance

---

## Troubleshooting Common Issues

### Issue: "Cannot connect to WordPress"
**Solution:** Check your environment variable
```bash
# In .env.local (local development)
NEXT_PUBLIC_WORDPRESS_URL=https://paulagonda.cybaxtech.com

# Don't use http://, use https://
# Don't add trailing slash
```

### Issue: "CORS errors when fetching from WordPress"
**Solution:** In WordPress, install and enable:
1. **WP CORS** plugin
2. Configure CORS headers in wp-config.php:
```php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
```

### Issue: "Domain doesn't resolve to Next.js app"
**Solution:** Update your DNS/domain settings:
1. Point `www.paulagonda.cybaxtech.com` to Vercel (or your hosting IP)
2. Update nameservers if using Vercel
3. Wait 24-48 hours for DNS propagation

---

## Next Steps

1. **Decide on hosting:** Vercel (recommended) vs. cPanel Node.js
2. **Update environment variable:** Replace `https://your-wordpress-site.com` with `https://paulagonda.cybaxtech.com`
3. **Test WordPress API:** Visit `https://paulagonda.cybaxtech.com/wp-json/wp/v2/posts`
4. **Deploy and test:** Verify all post types load correctly

---

## Quick Deploy to Vercel

```bash
# 1. Initialize Git
git init
git add .
git commit -m "Initial commit"

# 2. Push to GitHub
git remote add origin https://github.com/YOUR_USERNAME/paulagonda-portfolio.git
git push -u origin main

# 3. Go to vercel.com/new and import the GitHub repo
# 4. Add environment variable in Vercel dashboard:
#    NEXT_PUBLIC_WORDPRESS_URL = https://paulagonda.cybaxtech.com

# 5. Deploy (automatic)
```

That's it! Your site will be live on Vercel and pulling content from WordPress on cPanel.
