# Cache and Revalidation Guide

## Understanding Your Cache Issue

You're experiencing a caching delay because Next.js uses **Incremental Static Regeneration (ISR)** by default. This means pages are built once and reused until the cache expires.

## Current Configuration

- **Development**: 60 seconds (changed from 3600 for faster updates)
- **Production (Vercel)**: Should be higher to reduce unnecessary API calls

## How It Works

When you update a post in WordPress:
1. The Next.js site still serves the cached version for up to 60 seconds
2. After 60 seconds, Next.js checks WordPress for new data
3. If changes exist, it rebuilds the page in the background
4. Next request gets the updated version

## Cache Revalidation Options

### Option 1: Reduce Revalidation Time (Current)
For development, we've set it to 60 seconds. Changes appear within 1 minute.

**File**: `lib/wordpress.ts`
```typescript
next: { revalidate: 60 } // Check for updates every 60 seconds
```

### Option 2: No Caching (Development Only)
For fastest updates, set to 0 (no cache):

```typescript
next: { revalidate: 0 } // Fetch fresh data on every request
```

**Warning**: This is slow and makes many API calls. Only use during development.

### Option 3: Higher Cache for Production
For production (Vercel), use a higher value to reduce API usage:

```typescript
next: { revalidate: 3600 } // 1 hour cache
```

## How to Adjust Cache Times

1. Edit `/vercel/share/v0-project/lib/wordpress.ts`
2. Find the `revalidate` values in fetch calls
3. Change the number (in seconds):
   - `0` = Always fetch fresh (slow, many API calls)
   - `60` = 1 minute (good for development)
   - `300` = 5 minutes (balanced)
   - `3600` = 1 hour (good for production)
   - `86400` = 24 hours (for stable content)

## Important Environment Variable Issue

**Your .env.local is still using the placeholder URL!**

The debug logs show `WORDPRESS_URL: https://your-wordpress-site.com` instead of `https://paulagonda.cybaxtech.com`

### Fix This:

1. Open `.env.local` in your project root
2. Check this line:
   ```
   NEXT_PUBLIC_WORDPRESS_URL=https://paulagonda.cybaxtech.com
   ```
3. Make sure it's set to your actual WordPress URL
4. Restart your dev server (Ctrl+C, then `npm run dev`)

## Testing Your Setup

After fixing the environment variable:

1. Make a change to a post in WordPress
2. Wait up to 60 seconds (or your revalidate time)
3. Refresh your Next.js site
4. The change should appear

If it still doesn't appear, check:
- Is your WordPress REST API accessible? Visit `https://paulagonda.cybaxtech.com/wp-json/wp/v2/posts` in your browser
- Is CORS enabled in WordPress?
- Does the post type exist and have content?

## Production Recommendations

When deploying to Vercel:
- Use `revalidate: 3600` (1 hour) for most sites
- Use `revalidate: 300` (5 minutes) for frequently updated content
- Use `revalidate: 0` only for critical real-time content

## Advanced: On-Demand Revalidation

For instant updates without waiting, you can use webhooks with WordPress:

```typescript
// This requires a Server Action endpoint that calls revalidateTag()
// Set up a WordPress webhook to call: /api/revalidate?tag=posts
```

This is more complex but provides instant updates. Ask if you need help setting this up.
